// server.js
require('dotenv').config();
const express = require('express');
const cors = require('cors');
const mysql = require('mysql2/promise');
const multer = require('multer');
const fs = require('fs');
const path = require('path');
const ExcelJS = require('exceljs');
const archiver = require('archiver');

const app = express();
app.use(cors({
  origin: 'http://localhost:3000',
  credentials: true
}));
app.use(express.json());

// Storage for uploaded files
const UPLOAD_DIR = path.join(__dirname, 'uploads');
if (!fs.existsSync(UPLOAD_DIR)) fs.mkdirSync(UPLOAD_DIR);

// multer setup
const storage = multer.diskStorage({
  destination: (req, file, cb) => cb(null, UPLOAD_DIR),
  filename: (req, file, cb) => {
    // keep original filename but prefix with timestamp to avoid collisions
    const name = `${Date.now()}-${file.originalname}`;
    cb(null, name);
  }
});
const upload = multer({ storage });

// MySQL pool
const pool = mysql.createPool({
  host: process.env.DB_HOST || 'localhost',
  user: process.env.DB_USER || 'root',
  password: process.env.DB_PASS || '',
  database: process.env.DB_NAME || 'financeapp',
  waitForConnections: true,
  connectionLimit: 10,
  queueLimit: 0
});

// Allowed UMRN statuses to show on UI
const ALLOWED_UMRN_STATUSES = ['New','Rejected - Ready to resend','Modified'];

app.get('/api/loans', async (req, res) => {
  try {
    // optional query params: page, limit, search
    const [rows] = await pool.query(
      `SELECT * FROM loans WHERE UMRNStatus IN (?) ORDER BY \`index\` DESC`,
      [ALLOWED_UMRN_STATUSES]
    );
    res.json({ success: true, data: rows });
  } catch (err) {
    console.error('GET /api/loans error', err);
    res.status(500).json({ success: false, error: 'Server error' });
  }
});

app.get('/api/loans/:id', async (req, res) => {
  const id = req.params.id;
  try {
    const [loans] = await pool.query('SELECT * FROM loans WHERE Id = ?', [id]);
    if (!loans.length) return res.status(404).json({ success:false, error: 'Loan not found' });

    const loan = loans[0];
    const [docs] = await pool.query('SELECT * FROM documents WHERE ParentId = ?', [id]);

    res.json({ success: true, data: { loan, documents: docs } });
  } catch (err) {
    console.error('GET /api/loans/:id error', err);
    res.status(500).json({ success: false, error: 'Server error' });
  }
});

// upload document for a loan
app.post('/api/documents', upload.single('file'), async (req, res) => {
  try {
    const { parentId } = req.body;
    if (!req.file) return res.status(400).json({ success:false, error: 'No file uploaded' });
    if (!parentId) return res.status(400).json({ success:false, error: 'Missing parentId' });

    const filePath = req.file.filename; // relative path
    // generate an Id for doc; you can use uuid in production
    const docId = `doc-${Date.now()}`;

    await pool.query(
      'INSERT INTO documents (Id, Name, Link, ParentId) VALUES (?, ?, ?, ?)',
      [docId, req.file.originalname, filePath, parentId]
    );

    res.json({ success: true, data: { Id: docId, Name: req.file.originalname, Link: filePath, ParentId: parentId }});
  } catch (err) {
    console.error('POST /api/documents error', err);
    res.status(500).json({ success:false, error: 'Server error' });
  }
});

// download: update statuses and stream zip with Excel + related docs
app.post('/api/download', async (req, res) => {
  const { loanIds } = req.body; // expect array of loan Id strings
  if (!Array.isArray(loanIds) || loanIds.length === 0)
    return res.status(400).json({ success:false, error: 'loanIds required' });

  const connection = await pool.getConnection();
  try {
    await connection.beginTransaction();

    // 1) Fetch loans details
    const [loans] = await connection.query(
      `SELECT * FROM loans WHERE Id IN (?)`, [loanIds]
    );

    if (loans.length === 0) {
      await connection.rollback();
      return res.status(404).json({ success:false, error: 'No loans found' });
    }

    // 2) Update UMRNStatus -> 'Sent to Bank'
    await connection.query(
      `UPDATE loans SET UMRNStatus = ? WHERE Id IN (?)`,
      ['Sent to Bank', loanIds]
    );

    // 3) Get documents for these loans
    const [docs] = await connection.query(
      `SELECT * FROM documents WHERE ParentId IN (?)`,
      [loanIds]
    );

    // Commit DB changes (we update status before sending zip)
    await connection.commit();

    // 4) Create Excel workbook in memory
    const workbook = new ExcelJS.Workbook();
    const ws = workbook.addWorksheet('Loans');

    // choose columns to export; adapt as needed
    const columns = [
      { header: 'Id', key: 'Id' },
      { header: 'Name', key: 'Name' },
      { header: 'Dealer', key: 'Dealer' },
      { header: 'DealerFileNumber', key: 'DealerFileNumber' },
      { header: 'TotalPrice', key: 'TotalPrice' },
      { header: 'DisburseAmount', key: 'DisburseAmount' },
      { header: 'EMIAmount', key: 'EMIAmount' },
      { header: 'AgreementDate', key: 'AgreementDate' },
      { header: 'UMRNStatus', key: 'UMRNStatus' }
    ];
    ws.columns = columns;
    loans.forEach(l => {
      ws.addRow({
        Id: l.Id,
        Name: l.Name,
        Dealer: l.Dealer,
        DealerFileNumber: l.DealerFileNumber,
        TotalPrice: l.TotalPrice,
        DisburseAmount: l.DisburseAmount,
        EMIAmount: l.EMIAmount,
        AgreementDate: l.AgreementDate,
        UMRNStatus: 'Sent to Bank' // because we updated
      });
    });

    // 5) Stream ZIP to client
    res.setHeader('Content-Type', 'application/zip');
    res.setHeader('Content-Disposition', 'attachment; filename=loans_export.zip');

    const archive = archiver('zip', { zlib: { level: 9 }});
    archive.on('error', (err) => { throw err; });
    archive.pipe(res);

    // Add Excel as buffer
    const excelBuffer = await workbook.xlsx.writeBuffer();
    archive.append(excelBuffer, { name: 'loans.xlsx' });

    // Add each document file to zip, using path stored in documents.Link
    for (const d of docs) {
      const filePath = path.join(UPLOAD_DIR, d.Link);
      if (fs.existsSync(filePath)) {
        // to keep doc structure, put them under folder named by ParentId
        archive.file(filePath, { name: `documents/${d.ParentId}/${d.Name}` });
      } else {
        // If file missing, add a small text file explaining missing file
        archive.append(`File missing: ${d.Link}\nDocument record Id: ${d.Id}`, { name: `documents/${d.ParentId}/${d.Name}.missing.txt` });
      }
    }

    await archive.finalize();
    // response will end when archive finishes
  } catch (err) {
    await connection.rollback().catch(()=>{});
    console.error('/api/download error', err);
    res.status(500).json({ success:false, error: 'Server error during download' });
  } finally {
    connection.release();
  }
});

// Serve uploaded files for preview (be careful: production should protect files)
app.use('/uploads', express.static(UPLOAD_DIR));

const PORT = process.env.PORT || 4000;
app.listen(PORT, () => console.log(`Server listening on ${PORT}`));
